import { API_ENDPOINTS } from "@/lib/apiEndpoints";
import { Metadata } from "next";
import { notFound } from "next/navigation";
import { buildGuessPaperCategoryMeta } from "@/lib/seo/guessPaperMeta";
import GuessCoursesClient from "./GuessCoursesClient";

export const dynamic = "force-dynamic";

function stripHtml(html: string): string {
    return html.replace(/<[^>]*>?/gm, "").trim();
}

async function fetchCourseData(slug: string) {
    try {
        const res = await API_ENDPOINTS.GUESS_PAPERS.COURSE_DETAIL(slug);
        if (!res.ok) return null;
        const data = await res.json();
        return data; // { course, list }
    } catch {
        return null;
    }
}

export async function generateMetadata(
    { params }: { params: Promise<{ slug: string }> }
): Promise<Metadata> {
    const { slug } = await params;
    const data = await fetchCourseData(slug);

    if (!data || !data.course) {
        return {
            title: "Guess Papers Category",
        };
    }

    return buildGuessPaperCategoryMeta(data.course, slug);
}

export default async function GuessPaperCategoryPage({
    params
}: {
    params: Promise<{ slug: string }>;
}) {
    const { slug } = await params;
    const data = await fetchCourseData(slug);

    if (!data || !data.course) {
        notFound();
    }

    // Preserve the formatting/stripping logic from original client page
    // Original: setDescription(stripHtml(data.course.description));
    const description = stripHtml(data.course.description_guess_papers || "");

    return (
        <GuessCoursesClient
            courseCode={slug}
            headline={data.course.course_headline_guess_papers}
            description={description}
            breadcrumbHeadline={data.course.breadcrumb_headline_guess_papers}
            allAssignments={data.list || []}
        />
    );
}